unit ProcessClasses;
{*******************************************************************************
  ProcessChecker Demo
  Written by David Clegg, davidclegg@optusnet.com.au.

  This unit contains classes used to hold details for monitored and dependant
  processes.
*******************************************************************************}

interface

uses
  System.Runtime.Serialization, System.Collections;

type
  TProcess = class;

  /// <summary>
  /// Strongly typed collection of TProcess objects.
  /// </summary>
  [Serializable]
  TProcesses = class(CollectionBase)
  strict private
    function GetProcess(Index: integer): TProcess;
    procedure SetProcess(Index: integer; Value: TProcess);
  public
    procedure Insert(pIndex: integer; pProcess: TProcess);
    function Add(pProcess: TProcess): integer;
    procedure Remove(pProcess: TProcess);
    function Contains(pProcess: TProcess): boolean;
    function IndexOf(pProcess: TProcess): integer;
    property Process[Index: integer]: TProcess read GetProcess write SetProcess; default;
  end;

  [Serializable]
	/// <summary>
	/// Base class to contain details of processes to monitor
	/// </summary>
  TProcess = class
  strict private
    FPath: string;
    function GetName: string;
  public
    property Path: string read FPath write FPath;
    property Name: string read GetName;
    constructor Create(const pPath: string);
  end;

  /// <summary>
	/// Class to represent a process that is to be watched, and restarted if
	/// necessary.
	/// </summary>
	[Serializable]
  TWatchedProcess = class(TProcess)
  strict private
    FDependantProcesses: TProcesses;
    FCheckResponding: boolean;
  public
    property DependantProcesses: TProcesses read FDependantProcesses;
    property CheckResponding: boolean read FCheckResponding write FCheckResponding;
    constructor Create(const pPath: string);
  end;

  /// <summary>
	/// Class to represent a process that is a dependant of a watched process.
	/// If the watched process needs to be restarted, its dependants must be
	/// terminated first.
	/// </summary>
  [Serializable]
  TDependantProcess = class(TProcess)
  strict private
    FParent: TProcess;
  public
    property Parent: TProcess read FParent;
    constructor Create(const pPath: string; pParent: TProcess);
  end;

implementation

uses
  System.IO;

constructor TProcess.Create(const pPath: string);
begin
  inherited Create;
  FPath := pPath;
  // TODO: Add any constructor code here
end;

/// <summary>
/// Name of the process to be used for calls to Process.GetProcesses.
/// </summary>
function TProcess.GetName: string;
begin
  Result := System.IO.Path.GetFileNameWithoutExtension(FPath);
end;

constructor TWatchedProcess.Create(const pPath: string);
begin
  inherited Create(pPath);
  FDependantProcesses := TProcesses.Create;
end;

constructor TDependantProcess.Create(const pPath: string; pParent: TProcess);
begin
  inherited Create(pPath);
  FParent := pParent;
end;

function TProcesses.Add(pProcess: TProcess): integer;
begin
  Result := List.Add(pProcess);
end;

procedure TProcesses.Insert(pIndex: integer; pProcess: TProcess);
begin
  List.Insert(pIndex, pProcess);
end;

procedure TProcesses.Remove(pProcess: TProcess);
begin
  List.Remove(pProcess);
end;

function TProcesses.Contains(pProcess: TProcess): boolean;
begin
  Result := List.Contains(pProcess);
end;

function TProcesses.IndexOf(pProcess: TProcess): integer;
begin
  Result := List.IndexOf(pProcess);
end;

function TProcesses.GetProcess(Index: integer): TProcess;
begin
  Result := List[Index] as TProcess;
end;

procedure TProcesses.SetProcess(Index: integer; Value: TProcess);
begin
  List[Index] := Value;
end;

end.

